package com.expressotv.manager

import android.app.DatePickerDialog
import android.os.Bundle
import android.widget.ArrayAdapter
import android.widget.Toast
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.ViewModelProvider
import com.expressotv.manager.data.Client
import com.expressotv.manager.databinding.ActivityAddClientBinding
import com.expressotv.manager.viewmodel.ClientViewModel
import java.text.SimpleDateFormat
import java.util.*

class AddClientActivity : AppCompatActivity() {
    
    private lateinit var binding: ActivityAddClientBinding
    private lateinit var clientViewModel: ClientViewModel
    private var selectedDate: Calendar = Calendar.getInstance()
    
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityAddClientBinding.inflate(layoutInflater)
        setContentView(binding.root)
        
        setSupportActionBar(binding.toolbar)
        supportActionBar?.setDisplayHomeAsUpEnabled(true)
        supportActionBar?.title = "Adicionar Cliente"
        
        clientViewModel = ViewModelProvider(this)[ClientViewModel::class.java]
        
        setupSpinner()
        setupDatePicker()
        setupSaveButton()
    }
    
    private fun setupSpinner() {
        val planTypes = arrayOf("Básico", "Padrão", "Premium", "VIP")
        val adapter = ArrayAdapter(this, android.R.layout.simple_spinner_dropdown_item, planTypes)
        binding.spinnerPlanType.adapter = adapter
    }
    
    private fun setupDatePicker() {
        binding.btnSelectDate.setOnClickListener {
            val datePickerDialog = DatePickerDialog(
                this,
                { _, year, month, dayOfMonth ->
                    selectedDate.set(year, month, dayOfMonth)
                    updateDateButton()
                },
                selectedDate.get(Calendar.YEAR),
                selectedDate.get(Calendar.MONTH),
                selectedDate.get(Calendar.DAY_OF_MONTH)
            )
            datePickerDialog.datePicker.minDate = System.currentTimeMillis()
            datePickerDialog.show()
        }
        
        // Set default date to next month
        selectedDate.add(Calendar.MONTH, 1)
        updateDateButton()
    }
    
    private fun updateDateButton() {
        val sdf = SimpleDateFormat("dd/MM/yyyy", Locale.getDefault())
        binding.btnSelectDate.text = sdf.format(selectedDate.time)
    }
    
    private fun setupSaveButton() {
        binding.btnSaveClient.setOnClickListener {
            saveClient()
        }
    }
    
    private fun saveClient() {
        val name = binding.etClientName.text.toString().trim()
        val phone = binding.etClientPhone.text.toString().trim()
        val priceStr = binding.etMonthlyPrice.text.toString().trim()
        val notes = binding.etNotes.text.toString().trim()
        val planType = binding.spinnerPlanType.selectedItem.toString()
        
        // Validation
        if (name.isEmpty()) {
            binding.etClientName.error = "Nome é obrigatório"
            return
        }
        
        if (phone.isEmpty()) {
            binding.etClientPhone.error = "Telefone é obrigatório"
            return
        }
        
        if (priceStr.isEmpty()) {
            binding.etMonthlyPrice.error = "Preço é obrigatório"
            return
        }
        
        val price = try {
            priceStr.toDouble()
        } catch (e: NumberFormatException) {
            binding.etMonthlyPrice.error = "Preço inválido"
            return
        }
        
        val client = Client(
            name = name,
            phone = phone,
            planType = planType,
            expirationDate = selectedDate.timeInMillis,
            monthlyPrice = price,
            notes = notes,
            isActive = true
        )
        
        clientViewModel.insert(client)
        
        Toast.makeText(this, "Cliente adicionado com sucesso!", Toast.LENGTH_SHORT).show()
        finish()
    }
    
    override fun onSupportNavigateUp(): Boolean {
        finish()
        return true
    }
}
