package com.expressotv.manager

import android.app.DatePickerDialog
import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.widget.ArrayAdapter
import android.widget.Toast
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.lifecycleScope
import com.expressotv.manager.data.Client
import com.expressotv.manager.databinding.ActivityClientDetailsBinding
import com.expressotv.manager.utils.WhatsAppHelper
import com.expressotv.manager.viewmodel.ClientViewModel
import kotlinx.coroutines.launch
import java.text.SimpleDateFormat
import java.util.*

class ClientDetailsActivity : AppCompatActivity() {
    
    private lateinit var binding: ActivityClientDetailsBinding
    private lateinit var clientViewModel: ClientViewModel
    private var currentClient: Client? = null
    private var selectedDate: Calendar = Calendar.getInstance()
    
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityClientDetailsBinding.inflate(layoutInflater)
        setContentView(binding.root)
        
        setSupportActionBar(binding.toolbar)
        supportActionBar?.setDisplayHomeAsUpEnabled(true)
        supportActionBar?.title = "Detalhes do Cliente"
        
        clientViewModel = ViewModelProvider(this)[ClientViewModel::class.java]
        
        val clientId = intent.getIntExtra("CLIENT_ID", -1)
        if (clientId != -1) {
            loadClientData(clientId)
        }
        
        setupButtons()
    }
    
    private fun loadClientData(clientId: Int) {
        lifecycleScope.launch {
            currentClient = clientViewModel.getClientById(clientId)
            currentClient?.let { client ->
                displayClientData(client)
            }
        }
    }
    
    private fun displayClientData(client: Client) {
        binding.tvClientName.text = client.name
        binding.tvClientPhone.text = client.phone
        binding.tvPlanType.text = "Plano: ${client.planType}"
        binding.tvExpirationDate.text = "Vencimento: ${client.getFormattedExpirationDate()}"
        binding.tvMonthlyPrice.text = "Valor: R$ %.2f".format(client.monthlyPrice)
        binding.tvNotes.text = if (client.notes.isEmpty()) "Sem observações" else client.notes
        
        val daysUntil = client.getDaysUntilExpiration()
        when {
            client.isExpired() -> {
                binding.tvStatus.text = "Status: VENCIDO (${Math.abs(daysUntil)} dias)"
                binding.tvStatus.setTextColor(getColor(android.R.color.holo_red_dark))
            }
            client.isExpiringSoon() -> {
                binding.tvStatus.text = "Status: Vence em $daysUntil dias"
                binding.tvStatus.setTextColor(getColor(android.R.color.holo_orange_dark))
            }
            else -> {
                binding.tvStatus.text = "Status: Ativo ($daysUntil dias restantes)"
                binding.tvStatus.setTextColor(getColor(android.R.color.holo_green_dark))
            }
        }
    }
    
    private fun setupButtons() {
        binding.btnSendWhatsApp.setOnClickListener {
            currentClient?.let { client ->
                sendWhatsAppMessage(client)
            }
        }
        
        binding.btnCallClient.setOnClickListener {
            currentClient?.let { client ->
                callClient(client.phone)
            }
        }
        
        binding.btnRenewPlan.setOnClickListener {
            currentClient?.let { client ->
                showRenewDialog(client)
            }
        }
        
        binding.btnEditClient.setOnClickListener {
            currentClient?.let { client ->
                showEditDialog(client)
            }
        }
        
        binding.btnDeleteClient.setOnClickListener {
            currentClient?.let { client ->
                showDeleteConfirmation(client)
            }
        }
    }
    
    private fun sendWhatsAppMessage(client: Client) {
        val message = WhatsAppHelper.generateExpirationMessage(client)
        
        try {
            val intent = Intent(Intent.ACTION_VIEW)
            val url = "https://api.whatsapp.com/send?phone=55${client.phone}&text=${Uri.encode(message)}"
            intent.data = Uri.parse(url)
            startActivity(intent)
        } catch (e: Exception) {
            Toast.makeText(this, "WhatsApp não instalado", Toast.LENGTH_SHORT).show()
        }
    }
    
    private fun callClient(phone: String) {
        try {
            val intent = Intent(Intent.ACTION_DIAL)
            intent.data = Uri.parse("tel:$phone")
            startActivity(intent)
        } catch (e: Exception) {
            Toast.makeText(this, "Erro ao fazer chamada", Toast.LENGTH_SHORT).show()
        }
    }
    
    private fun showRenewDialog(client: Client) {
        selectedDate = Calendar.getInstance()
        selectedDate.timeInMillis = client.expirationDate
        selectedDate.add(Calendar.MONTH, 1)
        
        val dialog = AlertDialog.Builder(this)
            .setTitle("Renovar Plano")
            .setMessage("Selecione a nova data de vencimento")
            .setPositiveButton("Selecionar Data") { _, _ ->
                showDatePicker(client)
            }
            .setNegativeButton("Cancelar", null)
            .create()
        
        dialog.show()
    }
    
    private fun showDatePicker(client: Client) {
        DatePickerDialog(
            this,
            { _, year, month, dayOfMonth ->
                selectedDate.set(year, month, dayOfMonth)
                renewPlan(client)
            },
            selectedDate.get(Calendar.YEAR),
            selectedDate.get(Calendar.MONTH),
            selectedDate.get(Calendar.DAY_OF_MONTH)
        ).show()
    }
    
    private fun renewPlan(client: Client) {
        val updatedClient = client.copy(
            expirationDate = selectedDate.timeInMillis,
            isActive = true
        )
        
        clientViewModel.update(updatedClient)
        currentClient = updatedClient
        displayClientData(updatedClient)
        
        Toast.makeText(this, "Plano renovado com sucesso!", Toast.LENGTH_SHORT).show()
    }
    
    private fun showEditDialog(client: Client) {
        Toast.makeText(this, "Função de edição em desenvolvimento", Toast.LENGTH_SHORT).show()
    }
    
    private fun showDeleteConfirmation(client: Client) {
        AlertDialog.Builder(this)
            .setTitle("Excluir Cliente")
            .setMessage("Tem certeza que deseja excluir ${client.name}?")
            .setPositiveButton("Excluir") { _, _ ->
                clientViewModel.delete(client)
                Toast.makeText(this, "Cliente excluído", Toast.LENGTH_SHORT).show()
                finish()
            }
            .setNegativeButton("Cancelar", null)
            .show()
    }
    
    override fun onSupportNavigateUp(): Boolean {
        finish()
        return true
    }
}
