package com.expressotv.manager

import android.content.Intent
import android.os.Bundle
import android.view.Menu
import android.view.MenuItem
import android.widget.SearchView
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.ViewModelProvider
import androidx.recyclerview.widget.LinearLayoutManager
import com.expressotv.manager.adapter.ClientAdapter
import com.expressotv.manager.databinding.ActivityMainBinding
import com.expressotv.manager.utils.NotificationHelper
import com.expressotv.manager.viewmodel.ClientViewModel

class MainActivity : AppCompatActivity() {
    
    private lateinit var binding: ActivityMainBinding
    private lateinit var clientViewModel: ClientViewModel
    private lateinit var adapter: ClientAdapter
    
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityMainBinding.inflate(layoutInflater)
        setContentView(binding.root)
        
        setSupportActionBar(binding.toolbar)
        
        // Initialize notification channel
        NotificationHelper.createNotificationChannel(this)
        
        // Setup RecyclerView
        setupRecyclerView()
        
        // Setup ViewModel
        clientViewModel = ViewModelProvider(this)[ClientViewModel::class.java]
        
        // Observe clients
        clientViewModel.activeClients.observe(this) { clients ->
            adapter.submitList(clients)
            updateEmptyState(clients.isEmpty())
            
            // Update statistics
            val total = clients.size
            val expiring = clients.count { it.isExpiringSoon() }
            val expired = clients.count { it.isExpired() }
            
            binding.tvTotalClients.text = "Total: $total"
            binding.tvExpiringClients.text = "Vencendo: $expiring"
            binding.tvExpiredClients.text = "Vencidos: $expired"
        }
        
        // FAB click listener
        binding.fabAddClient.setOnClickListener {
            startActivity(Intent(this, AddClientActivity::class.java))
        }
    }
    
    private fun setupRecyclerView() {
        adapter = ClientAdapter(
            onItemClick = { client ->
                val intent = Intent(this, ClientDetailsActivity::class.java)
                intent.putExtra("CLIENT_ID", client.id)
                startActivity(intent)
            }
        )
        
        binding.recyclerViewClients.apply {
            layoutManager = LinearLayoutManager(this@MainActivity)
            adapter = this@MainActivity.adapter
            setHasFixedSize(true)
        }
    }
    
    private fun updateEmptyState(isEmpty: Boolean) {
        if (isEmpty) {
            binding.recyclerViewClients.visibility = android.view.View.GONE
            binding.emptyStateLayout.visibility = android.view.View.VISIBLE
        } else {
            binding.recyclerViewClients.visibility = android.view.View.VISIBLE
            binding.emptyStateLayout.visibility = android.view.View.GONE
        }
    }
    
    override fun onCreateOptionsMenu(menu: Menu): Boolean {
        menuInflater.inflate(R.menu.main_menu, menu)
        
        val searchItem = menu.findItem(R.id.action_search)
        val searchView = searchItem.actionView as SearchView
        
        searchView.setOnQueryTextListener(object : SearchView.OnQueryTextListener {
            override fun onQueryTextSubmit(query: String?): Boolean {
                return false
            }
            
            override fun onQueryTextChange(newText: String?): Boolean {
                if (newText.isNullOrEmpty()) {
                    clientViewModel.activeClients.observe(this@MainActivity) { clients ->
                        adapter.submitList(clients)
                    }
                } else {
                    clientViewModel.searchClients(newText).observe(this@MainActivity) { clients ->
                        adapter.submitList(clients)
                    }
                }
                return true
            }
        })
        
        return true
    }
    
    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        return when (item.itemId) {
            R.id.action_settings -> {
                // Open settings
                true
            }
            else -> super.onOptionsItemSelected(item)
        }
    }
}
