package com.expressotv.manager

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.WorkManager
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.expressotv.manager.data.ClientDatabase
import com.expressotv.manager.utils.NotificationHelper
import kotlinx.coroutines.runBlocking
import java.util.*
import java.util.concurrent.TimeUnit

class NotificationReceiver : BroadcastReceiver() {
    override fun onReceive(context: Context?, intent: Intent?) {
        context?.let {
            val workRequest = OneTimeWorkRequestBuilder<NotificationWorker>()
                .build()
            WorkManager.getInstance(it).enqueue(workRequest)
        }
    }
}

class NotificationWorker(
    private val context: Context,
    workerParams: WorkerParameters
) : Worker(context, workerParams) {
    
    override fun doWork(): Result {
        checkExpirations()
        scheduleNextCheck()
        return Result.success()
    }
    
    private fun checkExpirations() = runBlocking {
        val database = ClientDatabase.getDatabase(context)
        val dao = database.clientDao()
        
        // Check for clients expiring in the next 7 days
        val today = Calendar.getInstance().apply {
            set(Calendar.HOUR_OF_DAY, 0)
            set(Calendar.MINUTE, 0)
            set(Calendar.SECOND, 0)
            set(Calendar.MILLISECOND, 0)
        }
        
        val sevenDaysLater = Calendar.getInstance().apply {
            timeInMillis = today.timeInMillis
            add(Calendar.DAY_OF_MONTH, 7)
        }
        
        val clients = dao.getClientsExpiringBetween(
            today.timeInMillis,
            sevenDaysLater.timeInMillis
        )
        
        clients.forEach { client ->
            val daysUntil = client.getDaysUntilExpiration()
            if (daysUntil <= 7) {
                NotificationHelper.showExpirationNotification(
                    context,
                    client.name,
                    daysUntil
                )
            }
        }
    }
    
    private fun scheduleNextCheck() {
        val nextCheck = OneTimeWorkRequestBuilder<NotificationWorker>()
            .setInitialDelay(24, TimeUnit.HOURS)
            .build()
        
        WorkManager.getInstance(context).enqueue(nextCheck)
    }
}
