package com.expressotv.manager.adapter

import android.view.LayoutInflater
import android.view.ViewGroup
import androidx.recyclerview.widget.DiffUtil
import androidx.recyclerview.widget.ListAdapter
import androidx.recyclerview.widget.RecyclerView
import com.expressotv.manager.R
import com.expressotv.manager.data.Client
import com.expressotv.manager.databinding.ItemClientBinding

class ClientAdapter(
    private val onItemClick: (Client) -> Unit
) : ListAdapter<Client, ClientAdapter.ClientViewHolder>(ClientDiffCallback()) {
    
    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ClientViewHolder {
        val binding = ItemClientBinding.inflate(
            LayoutInflater.from(parent.context),
            parent,
            false
        )
        return ClientViewHolder(binding)
    }
    
    override fun onBindViewHolder(holder: ClientViewHolder, position: Int) {
        val client = getItem(position)
        holder.bind(client, onItemClick)
    }
    
    class ClientViewHolder(
        private val binding: ItemClientBinding
    ) : RecyclerView.ViewHolder(binding.root) {
        
        fun bind(client: Client, onItemClick: (Client) -> Unit) {
            binding.tvClientName.text = client.name
            binding.tvClientPhone.text = client.phone
            binding.tvPlanType.text = client.planType
            binding.tvExpirationDate.text = client.getFormattedExpirationDate()
            binding.tvPrice.text = "R$ %.2f".format(client.monthlyPrice)
            
            val daysUntil = client.getDaysUntilExpiration()
            when {
                client.isExpired() -> {
                    binding.tvDaysRemaining.text = "VENCIDO"
                    binding.tvDaysRemaining.setTextColor(
                        binding.root.context.getColor(android.R.color.holo_red_dark)
                    )
                    binding.cardView.setCardBackgroundColor(
                        binding.root.context.getColor(R.color.expired_background)
                    )
                }
                client.isExpiringSoon() -> {
                    binding.tvDaysRemaining.text = "$daysUntil dias"
                    binding.tvDaysRemaining.setTextColor(
                        binding.root.context.getColor(android.R.color.holo_orange_dark)
                    )
                    binding.cardView.setCardBackgroundColor(
                        binding.root.context.getColor(R.color.warning_background)
                    )
                }
                else -> {
                    binding.tvDaysRemaining.text = "$daysUntil dias"
                    binding.tvDaysRemaining.setTextColor(
                        binding.root.context.getColor(android.R.color.holo_green_dark)
                    )
                    binding.cardView.setCardBackgroundColor(
                        binding.root.context.getColor(R.color.active_background)
                    )
                }
            }
            
            binding.root.setOnClickListener {
                onItemClick(client)
            }
        }
    }
    
    class ClientDiffCallback : DiffUtil.ItemCallback<Client>() {
        override fun areItemsTheSame(oldItem: Client, newItem: Client): Boolean {
            return oldItem.id == newItem.id
        }
        
        override fun areContentsTheSame(oldItem: Client, newItem: Client): Boolean {
            return oldItem == newItem
        }
    }
}
